<?php
/**
 * CodeIgniter
 *
 * An open source application development framework for PHP
 *
 * This content is released under the MIT License (MIT)
 *
 * Copyright (c) 2019 - 2022, CodeIgniter Foundation
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 *
 * @package    CodeIgniter
 * @author    EllisLab Dev Team
 * @copyright    Copyright (c) 2008 - 2014, EllisLab, Inc. (https://ellislab.com/)
 * @copyright    Copyright (c) 2014 - 2019, British Columbia Institute of Technology (https://bcit.ca/)
 * @copyright    Copyright (c) 2019 - 2022, CodeIgniter Foundation (https://codeigniter.com/)
 * @license    https://opensource.org/licenses/MIT	MIT License
 * @link    https://codeigniter.com
 * @since    Version 3.0.0
 * @filesource
 */
defined('BASEPATH') or exit('No direct script access allowed');

/**
 * SQLite3 Forge Class
 *
 * @category    Database
 * @author    Andrey Andreev
 * @link    https://codeigniter.com/userguide3/database/
 */
class CI_DB_sqlite3_forge extends CI_DB_forge
{

    /**
     * UNSIGNED support
     *
     * @var    bool|array
     */
    protected $_unsigned = FALSE;

    /**
     * NULL value representation in CREATE/ALTER TABLE statements
     *
     * @var    string
     */
    protected $_null = 'NULL';

    // --------------------------------------------------------------------

    /**
     * Class constructor
     *
     * @param object    &$db Database object
     * @return    void
     */
    public function __construct(&$db)
    {
        parent::__construct($db);

        if (version_compare($this->db->version(), '3.3', '<')) {
            $this->_create_table_if = FALSE;
            $this->_drop_table_if = FALSE;
        }
    }

    // --------------------------------------------------------------------

    /**
     * Create database
     *
     * @param string $db_name
     * @return    bool
     */
    public function create_database($db_name)
    {
        // In SQLite, a database is created when you connect to the database.
        // We'll return TRUE so that an error isn't generated
        return TRUE;
    }

    // --------------------------------------------------------------------

    /**
     * Drop database
     *
     * @param string $db_name (ignored)
     * @return    bool
     */
    public function drop_database($db_name)
    {
        // In SQLite, a database is dropped when we delete a file
        if (file_exists($this->db->database)) {
            // We need to close the pseudo-connection first
            $this->db->close();
            if (!@unlink($this->db->database)) {
                return $this->db->db_debug ? $this->db->display_error('db_unable_to_drop') : FALSE;
            } elseif (!empty($this->db->data_cache['db_names'])) {
                $key = array_search(strtolower($this->db->database), array_map('strtolower', $this->db->data_cache['db_names']), TRUE);
                if ($key !== FALSE) {
                    unset($this->db->data_cache['db_names'][$key]);
                }
            }

            return TRUE;
        }

        return $this->db->db_debug ? $this->db->display_error('db_unable_to_drop') : FALSE;
    }

    // --------------------------------------------------------------------

    /**
     * ALTER TABLE
     *
     * @param string $alter_type ALTER type
     * @param string $table Table name
     * @param mixed $field Column definition
     * @return    string|string[]
     * @todo    implement drop_column(), modify_column()
     */
    protected function _alter_table($alter_type, $table, $field)
    {
        if ($alter_type === 'DROP' or $alter_type === 'CHANGE') {
            // drop_column():
            //	BEGIN TRANSACTION;
            //	CREATE TEMPORARY TABLE t1_backup(a,b);
            //	INSERT INTO t1_backup SELECT a,b FROM t1;
            //	DROP TABLE t1;
            //	CREATE TABLE t1(a,b);
            //	INSERT INTO t1 SELECT a,b FROM t1_backup;
            //	DROP TABLE t1_backup;
            //	COMMIT;

            return FALSE;
        }

        return parent::_alter_table($alter_type, $table, $field);
    }

    // --------------------------------------------------------------------

    /**
     * Process column
     *
     * @param array $field
     * @return    string
     */
    protected function _process_column($field)
    {
        return $this->db->escape_identifiers($field['name'])
            . ' ' . $field['type']
            . $field['auto_increment']
            . $field['null']
            . $field['unique']
            . $field['default'];
    }

    // --------------------------------------------------------------------

    /**
     * Field attribute TYPE
     *
     * Performs a data type mapping between different databases.
     *
     * @param array    &$attributes
     * @return    void
     */
    protected function _attr_type(&$attributes)
    {
        switch (strtoupper($attributes['TYPE'])) {
            case 'ENUM':
            case 'SET':
                $attributes['TYPE'] = 'TEXT';
                return;
            default:
                return;
        }
    }

    // --------------------------------------------------------------------

    /**
     * Field attribute AUTO_INCREMENT
     *
     * @param array    &$attributes
     * @param array    &$field
     * @return    void
     */
    protected function _attr_auto_increment(&$attributes, &$field)
    {
        if (!empty($attributes['AUTO_INCREMENT']) && $attributes['AUTO_INCREMENT'] === TRUE && stripos($field['type'], 'int') !== FALSE) {
            $field['type'] = 'INTEGER PRIMARY KEY';
            $field['default'] = '';
            $field['null'] = '';
            $field['unique'] = '';
            $field['auto_increment'] = ' AUTOINCREMENT';

            $this->primary_keys = array();
        }
    }

}
