<?php
/**
 * CodeIgniter
 *
 * An open source application development framework for PHP
 *
 * This content is released under the MIT License (MIT)
 *
 * Copyright (c) 2019 - 2022, CodeIgniter Foundation
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 *
 * @package    CodeIgniter
 * @author    EllisLab Dev Team
 * @copyright    Copyright (c) 2008 - 2014, EllisLab, Inc. (https://ellislab.com/)
 * @copyright    Copyright (c) 2014 - 2019, British Columbia Institute of Technology (https://bcit.ca/)
 * @copyright    Copyright (c) 2019 - 2022, CodeIgniter Foundation (https://codeigniter.com/)
 * @license    https://opensource.org/licenses/MIT	MIT License
 * @link    https://codeigniter.com
 * @since    Version 1.3.0
 * @filesource
 */
defined('BASEPATH') or exit('No direct script access allowed');

/**
 * ODBC Database Adapter Class
 *
 * Note: _DB is an extender class that the app controller
 * creates dynamically based on whether the query builder
 * class is being used or not.
 *
 * @package        CodeIgniter
 * @subpackage    Drivers
 * @category    Database
 * @author        EllisLab Dev Team
 * @link        https://codeigniter.com/userguide3/database/
 */
class CI_DB_odbc_driver extends CI_DB_driver
{

    /**
     * Database driver
     *
     * @var    string
     */
    public $dbdriver = 'odbc';

    /**
     * Database schema
     *
     * @var    string
     */
    public $schema = 'public';

    // --------------------------------------------------------------------

    /**
     * Identifier escape character
     *
     * Must be empty for ODBC.
     *
     * @var    string
     */
    protected $_escape_char = '';

    /**
     * ESCAPE statement string
     *
     * @var    string
     */
    protected $_like_escape_str = " {escape '%s'} ";

    /**
     * ORDER BY random keyword
     *
     * @var    array
     */
    protected $_random_keyword = array('RND()', 'RND(%d)');

    // --------------------------------------------------------------------

    /**
     * ODBC result ID resource returned from odbc_prepare()
     *
     * @var    resource
     */
    private $odbc_result;

    /**
     * Values to use with odbc_execute() for prepared statements
     *
     * @var    array
     */
    private $binds = array();

    // --------------------------------------------------------------------

    /**
     * Class constructor
     *
     * @param array $params
     * @return    void
     */
    public function __construct($params)
    {
        parent::__construct($params);

        // Legacy support for DSN in the hostname field
        if (empty($this->dsn)) {
            $this->dsn = $this->hostname;
        }
    }

    // --------------------------------------------------------------------

    /**
     * Non-persistent database connection
     *
     * @param bool $persistent
     * @return    resource
     */
    public function db_connect($persistent = FALSE)
    {
        return ($persistent === TRUE)
            ? odbc_pconnect($this->dsn, $this->username, $this->password)
            : odbc_connect($this->dsn, $this->username, $this->password);
    }

    // --------------------------------------------------------------------

    /**
     * Compile Bindings
     *
     * @param string $sql SQL statement
     * @param array $binds An array of values to bind
     * @return    string
     */
    public function compile_binds($sql, $binds)
    {
        if (empty($binds) or empty($this->bind_marker) or strpos($sql, $this->bind_marker) === FALSE) {
            return $sql;
        } elseif (!is_array($binds)) {
            $binds = array($binds);
            $bind_count = 1;
        } else {
            // Make sure we're using numeric keys
            $binds = array_values($binds);
            $bind_count = count($binds);
        }

        // We'll need the marker length later
        $ml = strlen($this->bind_marker);

        // Make sure not to replace a chunk inside a string that happens to match the bind marker
        if ($c = preg_match_all("/'[^']*'|\"[^\"]*\"/i", $sql, $matches)) {
            $c = preg_match_all('/' . preg_quote($this->bind_marker, '/') . '/i',
                str_replace($matches[0],
                    str_replace($this->bind_marker, str_repeat(' ', $ml), $matches[0]),
                    $sql, $c),
                $matches, PREG_OFFSET_CAPTURE);

            // Bind values' count must match the count of markers in the query
            if ($bind_count !== $c) {
                return $sql;
            }
        } elseif (($c = preg_match_all('/' . preg_quote($this->bind_marker, '/') . '/i', $sql, $matches, PREG_OFFSET_CAPTURE)) !== $bind_count) {
            return $sql;
        }

        if ($this->bind_marker !== '?') {
            do {
                $c--;
                $sql = substr_replace($sql, '?', $matches[0][$c][1], $ml);
            } while ($c !== 0);
        }

        if (FALSE !== ($this->odbc_result = odbc_prepare($this->conn_id, $sql))) {
            $this->binds = array_values($binds);
        }

        return $sql;
    }

    // --------------------------------------------------------------------

    /**
     * Execute the query
     *
     * @param string $sql an SQL query
     * @return    resource
     */
    protected function _execute($sql)
    {
        if (!isset($this->odbc_result)) {
            return odbc_exec($this->conn_id, $sql);
        } elseif ($this->odbc_result === FALSE) {
            return FALSE;
        }

        if (TRUE === ($success = odbc_execute($this->odbc_result, $this->binds))) {
            // For queries that return result sets, return the result_id resource on success
            $this->is_write_type($sql) or $success = $this->odbc_result;
        }

        $this->odbc_result = NULL;
        $this->binds = array();

        return $success;
    }

    // --------------------------------------------------------------------

    /**
     * Begin Transaction
     *
     * @return    bool
     */
    protected function _trans_begin()
    {
        return odbc_autocommit($this->conn_id, FALSE);
    }

    // --------------------------------------------------------------------

    /**
     * Commit Transaction
     *
     * @return    bool
     */
    protected function _trans_commit()
    {
        if (odbc_commit($this->conn_id)) {
            odbc_autocommit($this->conn_id, TRUE);
            return TRUE;
        }

        return FALSE;
    }

    // --------------------------------------------------------------------

    /**
     * Rollback Transaction
     *
     * @return    bool
     */
    protected function _trans_rollback()
    {
        if (odbc_rollback($this->conn_id)) {
            odbc_autocommit($this->conn_id, TRUE);
            return TRUE;
        }

        return FALSE;
    }

    // --------------------------------------------------------------------

    /**
     * Determines if a query is a "write" type.
     *
     * @param string    An SQL query string
     * @return    bool
     */
    public function is_write_type($sql)
    {
        if (preg_match('#^(INSERT|UPDATE).*RETURNING\s.+(\,\s?.+)*$#is', $sql)) {
            return FALSE;
        }

        return parent::is_write_type($sql);
    }

    // --------------------------------------------------------------------

    /**
     * Platform-dependent string escape
     *
     * @param string
     * @return    string
     */
    protected function _escape_str($str)
    {
        $this->display_error('db_unsupported_feature');
    }

    // --------------------------------------------------------------------

    /**
     * Affected Rows
     *
     * @return    int
     */
    public function affected_rows()
    {
        return odbc_num_rows($this->result_id);
    }

    // --------------------------------------------------------------------

    /**
     * Insert ID
     *
     * @return    bool
     */
    public function insert_id()
    {
        return ($this->db_debug) ? $this->display_error('db_unsupported_feature') : FALSE;
    }

    // --------------------------------------------------------------------

    /**
     * Show table query
     *
     * Generates a platform-specific query string so that the table names can be fetched
     *
     * @param bool $prefix_limit
     * @return    string
     */
    protected function _list_tables($prefix_limit = FALSE)
    {
        $sql = "SELECT table_name FROM information_schema.tables WHERE table_schema = '" . $this->schema . "'";

        if ($prefix_limit !== FALSE && $this->dbprefix !== '') {
            return $sql . " AND table_name LIKE '" . $this->escape_like_str($this->dbprefix) . "%' "
                . sprintf($this->_like_escape_str, $this->_like_escape_chr);
        }

        return $sql;
    }

    // --------------------------------------------------------------------

    /**
     * Show column query
     *
     * Generates a platform-specific query string so that the column names can be fetched
     *
     * @param string $table
     * @return    string
     */
    protected function _list_columns($table = '')
    {
        return 'SHOW COLUMNS FROM ' . $table;
    }

    // --------------------------------------------------------------------

    /**
     * Field data query
     *
     * Generates a platform-specific query so that the column data can be retrieved
     *
     * @param string $table
     * @return    string
     */
    protected function _field_data($table)
    {
        return 'SELECT TOP 1 FROM ' . $table;
    }

    // --------------------------------------------------------------------

    /**
     * Error
     *
     * Returns an array containing code and message of the last
     * database error that has occurred.
     *
     * @return    array
     */
    public function error()
    {
        return array('code' => odbc_error($this->conn_id), 'message' => odbc_errormsg($this->conn_id));
    }

    // --------------------------------------------------------------------

    /**
     * Close DB Connection
     *
     * @return    void
     */
    protected function _close()
    {
        odbc_close($this->conn_id);
    }
}
